// ==================== КОНФИГУРАЦИЯ ====================
const SERVER_URL = "https://id-manager-t.duckdns.org";
const SITE_NAME = window.location.hostname.replace(/^(www\.|mobi\.)/, "");

// КОРРЕКТНЫЕ ЭНДПОИНТЫ ДЛЯ МОДУЛЬНОГО СЕРВЕРА
const API_ENDPOINTS = {
  // Основные эндпоинты для расширения
  GET_ID: (id, site) => `${SERVER_URL}/api/ids/get-id/${encodeURIComponent(id)}?site=${encodeURIComponent(site)}`,
  ADD_ID: `${SERVER_URL}/api/ids/add-id`,
  
  // Парсеры - ПРАВИЛЬНЫЙ ПУТЬ!
  PARSERS: `${SERVER_URL}/api/parsers/parsers`,
  PARSERS_EXPORT: `${SERVER_URL}/api/parsers/parsers/export`,
  
  // Настройки сайта - ПРАВИЛЬНЫЙ ПУТЬ!
  SITE_SETTINGS: `${SERVER_URL}/api/sites/site-settings`,
  
  // Группы сайтов - ПРАВИЛЬНЫЙ ПУТЬ!
  SITE_GROUPS: `${SERVER_URL}/api/sites/site-groups`,
  
  // Пакетная проверка
  BULK_CHECK: `${SERVER_URL}/api/ids/bulk-check-ids`,
  
  // Ping
  PING: `${SERVER_URL}/api/ids/ping`,
  PING_FAST: `${SERVER_URL}/api/ids/ping-fast`
};

// ==================== ГЛОБАЛЬНЫЕ ПЕРЕМЕННЫЕ ====================
// ДИНАМИЧЕСКИЕ СЕЛЕКТОРЫ
let DYNAMIC_SELECTORS = '';

// Кэш настроек сайта и групп
let siteSettings = {
    minDuplicates: 2,
    minDate: "2025-09-22",
    useMinDuplicates: true,
    useMinDate: true,
    siteGroups: []
};

// Кэш парсеров
let siteParsers = [];
let lastParsersLoad = 0;
// Кешируем парсеры дольше, чтобы снизить частоту запросов к серверу
const PARSERS_CACHE_TIME = 5 * 60 * 1000; // 5 минут

// КЭШ ДЛЯ ЗАПИСЕЙ И СОСТОЯНИЕ
const recordsCache = new Map();
const updateInProgress = new Set();
let lastSettingsLoad = 0;  // <-- ДОБАВЬТЕ ЭТУ ПЕРЕМЕННУЮ!
// Настройки тоже кешируем дольше (можно уменьшить при необходимости)
const SETTINGS_CACHE_TIME = 2 * 60 * 1000; // 2 минуты

// URL для отслеживания SPA навигации
let currentUrl = window.location.href;

// Наблюдатель
let observer = null;

// ==================== ИСПРАВЛЕННЫЕ ФУНКЦИИ ====================

// Загрузка парсеров - ИСПРАВЛЕННЫЙ ПУТЬ
// Загрузка парсеров - ОПТИМИЗИРОВАННЫЙ ПУТЬ
async function loadSiteParsers() {
    const now = Date.now();
    if (now - lastParsersLoad < PARSERS_CACHE_TIME) {
        return siteParsers;
    }
    
    try {
        // console.log(`🔍 Загрузка парсеров для: ${SITE_NAME}`);
        
        // Приоритет у нового эндпоинта экспорта
        let parsers = [];
        
        try {
            const res = await fetch(`${API_ENDPOINTS.PARSERS_EXPORT}?site=${encodeURIComponent(SITE_NAME)}`);
            if (res.ok) {
                const data = await res.json();
                if (data.success && data.parsers) {
                    parsers = data.parsers.filter(p => p.enabled !== false);
                    // console.log(`✅ Загружено ${parsers.length} парсеров через экспорт`);
                }
            }
        } catch (exportError) {
            // console.log('📡 Экспорт парсеров не доступен, пробуем обычный эндпоинт...');
        }
        
        // Fallback если экспорт не сработал
        if (parsers.length === 0) {
            const res = await fetch(`${API_ENDPOINTS.PARSERS}?site=${encodeURIComponent(SITE_NAME)}`);
            if (res.ok) {
                const serverParsers = await res.json();
                parsers = Array.isArray(serverParsers) ? serverParsers.filter(p => p.is_active !== false) : [];
                // console.log(`✅ Загружено ${parsers.length} парсеров (обычный эндпоинт)`);
            }
        }
        
        siteParsers = parsers;
        lastParsersLoad = now;
        updateDynamicSelectors();
        
        // Уведомляем popup
        chrome.runtime.sendMessage({
            action: 'parsersUpdated',
            parsers: siteParsers,
            parsersCount: siteParsers.length,
            selectorsCount: Array.from(new Set(siteParsers.flatMap(p => p.selectors || []))).length,
            site: SITE_NAME,
            groups: siteSettings.siteGroups || []
        });
        
        return siteParsers;
        
    } catch (e) {
        console.warn("❌ Ошибка загрузки парсеров:", e);
        siteParsers = [];
        return [];
    }
}


// Загрузка настроек сайта - ИСПРАВЛЕННЫЙ ПУТЬ
async function loadSiteSettings() {
    const now = Date.now();
    if (now - lastSettingsLoad < SETTINGS_CACHE_TIME) {
        return;
    }
    
    try {
        // ИСПРАВЛЕННЫЙ ПУТЬ!
        const res = await fetch(`${API_ENDPOINTS.SITE_SETTINGS}?site=${encodeURIComponent(SITE_NAME)}`);
        
        if (res.ok) {
            const settings = await res.json();
            
            siteSettings = {
                minDuplicates: settings.minDuplicates == null ? 2 : Number(settings.minDuplicates),
                minDate: settings.minDate || getDefaultMinDate(),
                useMinDuplicates: settings.useMinDuplicates === false ? false : true,
                useMinDate: settings.useMinDate === false ? false : true,
                siteGroups: settings.siteGroups || []
            };
            
            chrome.storage.local.set({ currentSiteSettings: siteSettings });
            lastSettingsLoad = now;
            
            // Уведомляем popup об обновлении
            chrome.runtime.sendMessage({
                action: 'settingsUpdated',
                settings: siteSettings,
                site: SITE_NAME
            });
        } else {
            console.warn(`❌ Ошибка загрузки настроек: HTTP ${res.status}`);
            // Устанавливаем настройки по умолчанию
            siteSettings = {
                minDuplicates: 2,
                minDate: getDefaultMinDate(),
                useMinDuplicates: true,
                useMinDate: true,
                siteGroups: []
            };
        }
    } catch (e) {
        console.warn("❌ Ошибка загрузки настроек:", e);
        // Устанавливаем настройки по умолчанию при ошибке
        siteSettings = {
            minDuplicates: 2,
            minDate: getDefaultMinDate(),
            useMinDuplicates: true,
            useMinDate: true,
            siteGroups: []
        };
    }
}

function getDefaultMinDate() {
    const date = new Date();
    date.setDate(date.getDate() - 30);
    return date.toISOString().split('T')[0];
}

// ==================== ИСПРАВЛЕННАЯ ФУНКЦИЯ ДЛЯ РАБОТЫ С ПУСТЫМИ СЕЛЕКТОРАМИ ====================

// Функция для получения текущих селекторов (с защитой)
function getCurrentSelectors() {
    if (!DYNAMIC_SELECTORS || DYNAMIC_SELECTORS.trim() === '') {
        // Возвращаем безопасный селектор, который ничего не найдет
        return '.universal-id-no-elements-found';
    }
    return DYNAMIC_SELECTORS;
}
// ==================== ФУНКЦИИ СЕЛЕКТОРОВ ====================

// Функция для обновления селекторов из парсеров
function updateDynamicSelectors() {
    if (!siteParsers || siteParsers.length === 0) {
        console.log('ℹ️ Нет парсеров для обновления селекторов');
        DYNAMIC_SELECTORS = ''; // Очищаем селекторы если нет парсеров
        return;
    }
    
    const allSelectors = new Set();
    
    // ДОБАВЛЯЕМ ТОЛЬКО СЕЛЕКТОРЫ ИЗ ПАРСЕРОВ - без базовых
    siteParsers.forEach(parser => {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            parser.selectors.forEach(selector => {
                if (selector && selector.trim() && selector.length > 2) {
                    allSelectors.add(selector.trim());
                }
            });
        }
    });
    
    // Обновляем глобальную переменную
    DYNAMIC_SELECTORS = Array.from(allSelectors).join(', ');
    
    // console.log(`🎯 Обновлены динамические селекторы из парсеров: ${allSelectors.size} селекторов`, Array.from(allSelectors));
    
    // Уведомляем об обновлении
    chrome.runtime.sendMessage({
        action: 'selectorsUpdated',
        selectorsCount: allSelectors.size,
        selectors: Array.from(allSelectors).slice(0, 10)
    });
}

// Функция проверки соответствия элемента селекторам парсеров
function elementMatchesAnyParserSelector(element) {
    if (!siteParsers || siteParsers.length === 0) {
        return true; // Если парсеров нет, пропускаем все элементы
    }
    
    for (const parser of siteParsers) {
        if (parser.selectors && Array.isArray(parser.selectors)) {
            for (const selector of parser.selectors) {
                try {
                    if (selector && element.matches && element.matches(selector)) {
                        // console.log(`🎯 Элемент соответствует селектору парсера "${parser.name}": ${selector}`);
                        return true;
                    }
                } catch (selectorError) {
                    console.warn(`❌ Ошибка в селекторе "${selector}" парсера "${parser.name}":`, selectorError);
                }
            }
        }
    }
    
    return false;
}

// ==================== ПАРСЕРЫ ID ====================

function extractIDFromElement(el) {
    if (!el || !el.nodeType) {
        return null;
    }
    
    // ИСПОЛЬЗУЕМ ТОЛЬКО ПАРСЕРЫ - никаких стандартных методов
    if (siteParsers.length > 0) {
        const id = extractIDWithParsers(el);
        if (id && isValidId(id)) {
            // console.log(`🎯 Парсер извлёк ВАЛИДНЫЙ ID: ${id}`);
            return id;
        }
    }
    
    return null;
}

// Улучшенная функция парсеров
function extractIDWithParsers(element) {
    const html = element.outerHTML;
    let foundId = null;
    
    if (!siteParsers || siteParsers.length === 0) {
        return null;
    }
    
    for (const parser of siteParsers) {
        if (foundId) break;
        
        try {
            // console.log(`🔍 Применяем парсер "${parser.name}" для сайта ${parser.site}`);
            
            if (parser.pattern && parser.pattern.length > 2) {
                try {
                    const regex = new RegExp(parser.pattern, 'gi');
                    regex.lastIndex = 0; // Сбрасываем позицию
                    
                    const match = regex.exec(html);
                    if (match) {
                        // Используем первую группу захвата или всё совпадение
                        foundId = match[1] || match[0];
                        if (foundId) {
                            foundId = foundId.trim();
                            // console.log(`✅ Парсер "${parser.name}" извлёк: ${foundId}`);
                            break;
                        }
                    }
                } catch (regexError) {
                    console.warn(`❌ Ошибка regex в парсере "${parser.name}":`, regexError.message);
                    continue;
                }
            }
        } catch (error) {
            console.error(`❌ Общая ошибка в парсере "${parser.name}":`, error);
        }
    }
    
    return foundId;
}

// Функция проверки валидности ID
function isValidId(id) {
    if (!id || typeof id !== 'string') return false;
    
    const cleanId = id.trim();
    
    // Сначала пропустим явно мусорные значения
    const invalidPatterns = [
        /^factfile$/i,
        /^testresult$/i, 
        /^partnersuggestions$/i,
        /^matches$/i,
        /^profile$/i,
        /^partner$/i,
        /^https?:\/\//,
        /^\/\//,
        /^[.#]/,
        /^[\s\S]{1,2}$/ // Очень короткие (1-2 символа)
    ];
    
    for (const pattern of invalidPatterns) {
        if (pattern && pattern.test(cleanId)) {
            console.log(`🚫 Отсеян мусорный ID: ${cleanId}`);
            return false;
        }
    }
    
    // НОВЫЕ ПРАВИЛА ВАЛИДАЦИИ - более гибкие!
    const validPatterns = [
        /^[A-Z0-9]{8,12}$/, // Parship ID (PSPYD8M2)
        /^[a-zA-Z0-9_-]{4,30}$/, // Общие alphanumeric ID - ИЗМЕНИЛИ с 6 на 4!
        /^\d{4,20}$/, // Числовые ID
        /^[a-f0-9]{8}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{4}-[a-f0-9]{12}$/i, // UUID
        /^[a-zA-Z0-9]{4,35}$/, // Длинные ID - ИЗМЕНИЛИ с 15 на 4!
        /^[a-z0-9]{3,10}$/i, // Короткие буквенно-цифровые (типа x0gib, squcy)
        /^[a-z]{4,10}$/i // Чисто буквенные 4-10 символов
    ];
    
    // Проверяем соответствие
    const isValid = validPatterns.some(pattern => pattern.test(cleanId));
    
    // ДОПОЛНИТЕЛЬНО: проверка на наличие хоть одного числа или буквы
    const hasLetter = /[a-zA-Z]/.test(cleanId);
    const hasNumber = /\d/.test(cleanId);
    const hasSpecial = /[_-]/.test(cleanId);
    
    const hasValidChars = hasLetter || hasNumber || hasSpecial;
    
    const finalIsValid = isValid && hasValidChars && cleanId.length >= 3;
    
    // console.log(`🔍 Проверка ID "${cleanId}":`, {
    //     length: cleanId.length,
    //     hasLetter,
    //     hasNumber,
    //     hasSpecial,
    //     matchesPattern: isValid,
    //     final: finalIsValid ? 'VALID' : 'INVALID'
    // });
    
    return finalIsValid;
}

// ==================== ПЕРЕСКАНИРОВАНИЕ СТРАНИЦЫ ====================

// Функция пересканирования страницы с новыми селекторами - ТОЛЬКО ДЛЯ НОВЫХ ЭЛЕМЕНТОВ
function rescanPageWithNewSelectors() {
    // console.log('🔄 Пересканирование страницы с обновленными селекторами');
    
    // НЕ удаляем старые контролы! Только сбрасываем флаги для НОВЫХ элементов
    const currentSelectors = getCurrentSelectors();
    const allElements = document.querySelectorAll(currentSelectors);
    
    // console.log(`🎯 Найдено элементов по новым селекторам: ${allElements.length}`);
    
    // Обрабатываем только элементы без флага enhanced
    allElements.forEach(el => {
        if (!el._universalEnhanced) {
            enhanceCard(el);
        }
    });
}

// "Тихое" обновление статуса без пересоздания контролов
async function quietUpdateCardStatuses() {
    const currentSelectors = getCurrentSelectors();
    const cards = document.querySelectorAll(currentSelectors);
    
    // console.log(`🔍 Тихое обновление статусов: ${cards.length} карточек`);
    
    for (const el of cards) {
        if (el._universalEnhanced) {
            const id = extractIDFromElement(el);
            if (id) {
                await updateCardStatus(id, el);
            }
        }
    }
}


// ПАССИВНОЕ сканирование - только поиск без отправки
function scanPageWithParsers() {
    if (siteParsers.length === 0) {
        // console.log("ℹ️ Нет активных парсеров для сканирования");
        return;
    }
    
    try {
        const pageHtml = document.documentElement.outerHTML;
        const foundIds = new Set();
        
        // console.log(`🔍 Сканирование страницы с ${siteParsers.length} парсерами (только поиск)`);
        
        // Применяем парсеры для поиска ID по всей странице
        for (const parser of siteParsers) {
            try {
                if (parser.pattern && parser.pattern.length > 2) {
                    try {
                        const regex = new RegExp(parser.pattern, 'gi');
                        let match;
                        regex.lastIndex = 0;
                        
                        while ((match = regex.exec(pageHtml)) !== null) {
                            let extractedId = match[1] || match[0];
                            
                            if (extractedId && extractedId.length >= 3 && isValidId(extractedId)) {
                                extractedId = extractedId.trim();
                                // console.log(`✅ Сканирование: парсер "${parser.name}" нашёл ID: ${extractedId}`);
                                foundIds.add(extractedId);
                            }
                        }
                    } catch (regexError) {
                        console.warn(`❌ Ошибка regex в сканировании (${parser.name}):`, regexError);
                    }
                }
            } catch (error) {
                console.error(`❌ Ошибка в парсере ${parser.name} при сканировании:`, error);
            }
        }
        
        // ТОЛЬКО ЛОГИРОВАНИЕ - НИКАКОЙ ОТПРАВКИ
        if (foundIds.size > 0) {
            // console.log(`🎯 Сканирование найдено ${foundIds.size} ID (не отправляются автоматически):`, Array.from(foundIds));
            
            // Можно показать уведомление о найденных ID, но не отправлять
            showNotification(`🔍 Найдено ${foundIds.size} ID на странице`);
        }
        
    } catch (error) {
        console.error('❌ Ошибка сканирования:', error);
    }
}

// ==================== ЛОГИКА ПОДСВЕТКИ ====================

function shouldHighlight(records) {
    if (!Array.isArray(records) || records.length === 0) {
        // console.log('[shouldHighlight] Нет записей для подсветки');
        return false;
    }

    const count = records.length;
    const useDup = siteSettings.useMinDuplicates;
    const useDate = siteSettings.useMinDate;

    // console.log('[shouldHighlight] siteSettings:', siteSettings);
    // console.log('[shouldHighlight] records:', records);

    if (!useDup && !useDate) {
        // console.log('[shouldHighlight] Подсветка всегда (оба выключены)');
        return true;
    }

    let meetsDuplicates = false;
    let meetsDate = false;

    if (useDup) {
        meetsDuplicates = count >= siteSettings.minDuplicates;
        // console.log(`[shouldHighlight] meetsDuplicates: ${meetsDuplicates} (count=${count}, minDuplicates=${siteSettings.minDuplicates})`);
    }

    if (useDate) {
        const thresholdDate = new Date(siteSettings.minDate);
        meetsDate = records.some(r => {
            if (!r.created_at) return false;
            const recordDate = new Date(r.created_at);
            const valid = !isNaN(recordDate) && recordDate >= thresholdDate;
            // console.log(`[shouldHighlight] Проверка даты: created_at=${r.created_at}, recordDate=${recordDate}, thresholdDate=${thresholdDate}, valid=${valid}`);
            return valid;
        });
        // console.log(`[shouldHighlight] meetsDate: ${meetsDate} (minDate=${siteSettings.minDate})`);
    }

    if (useDup && !useDate) {
        // console.log(`[shouldHighlight] Итог meetsDuplicates: ${meetsDuplicates}`);
        return meetsDuplicates;
    } else if (!useDup && useDate) {
        // console.log(`[shouldHighlight] Итог meetsDate: ${meetsDate}`);
        return meetsDate;
    } else if (useDup && useDate) {
        // console.log(`[shouldHighlight] Итог meetsDuplicates || meetsDate: ${meetsDuplicates || meetsDate}`);
        return meetsDuplicates || meetsDate;
    }

    // console.log('[shouldHighlight] Итог: false');
    return false;
}

function applyHighlight(element, shouldHighlight) {
    if (shouldHighlight) {
        element.style.transition = "background-color 0.3s ease";
        element.style.backgroundColor = "rgba(255,180,180,0.35)";
        element.style.outline = "2px solid #ff3366";
        element.style.outlineOffset = "1px";
        element.style.boxShadow = "0 0 10px rgba(255, 51, 102, 0.5)";
    } else {
        element.style.backgroundColor = "";
        element.style.outline = "";
        element.style.boxShadow = "";
    }
}

// ==================== УПРОЩЁННЫЕ API ЗАПРОСЫ ====================

async function getApiKey() {
    return new Promise((resolve) => {
        chrome.storage.local.get(["apiKey", "my_api_key"], (res) => {
            resolve(res.apiKey || res.my_api_key || null);
        });
    });
}

// ОБНОВЛЕННАЯ функция проверки записей с учетом групп

// ==================== BULK API ДЛЯ ID ====================
// Пакетная проверка ID
async function bulkCheckIds(ids) {
    if (!Array.isArray(ids) || ids.length === 0) return {};
    try {
        const payload = { ids, site: SITE_NAME };
        // Используем таймаут и AbortController, чтобы не висеть и не накапливать запросы
        const controller = new AbortController();
        const timeout = setTimeout(() => controller.abort(), 8000);
        // console.log('[bulkCheckIds] Отправка запроса:', JSON.stringify(payload));
        const res = await fetch(API_ENDPOINTS.BULK_CHECK, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload),
            signal: controller.signal
        });
        clearTimeout(timeout);
        const text = await res.text();
        // console.log('[bulkCheckIds] Ответ сервера:', text);
        let data;
        try {
            data = JSON.parse(text);
        } catch (parseErr) {
            console.error('[bulkCheckIds] Ошибка парсинга ответа:', parseErr);
            return {};
        }
        // Возвращаем либо data.results, либо сам data
        if (data && typeof data === 'object') {
            if (data.results && typeof data.results === 'object') {
                return data.results;
            }
            return data;
        }
        return {};
    } catch (e) {
        console.error("Ошибка пакетной проверки ID:", e);
    }
    return {};
}

// Пакетное добавление ID
async function bulkAddIds(ids, apiKey) {
    if (!Array.isArray(ids) || ids.length === 0) return false;
    try {
        const payload = { ids, apiKey, site: SITE_NAME };
        // console.log('[bulkAddIds] Отправка запроса:', JSON.stringify(payload));
        const res = await fetch(`${SERVER_URL}/api/ids/bulk-add`, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(payload)
        });
        const text = await res.text();
        // console.log('[bulkAddIds] Ответ сервера:', text);
        let data;
        try {
            data = JSON.parse(text);
        } catch (parseErr) {
            console.error('[bulkAddIds] Ошибка парсинга ответа:', parseErr);
            return false;
        }
        return data.success === true;
    } catch (e) {
        console.error("Ошибка пакетной отправки ID:", e);
    }
    return false;
}

// Одиночная проверка через bulkCheckIds
async function fetchRecordsForId(id) {
    const CACHE_TIME = 3000;
    const now = Date.now();
    if (recordsCache.has(id)) {
        const cached = recordsCache.get(id);
        if (now - cached.timestamp < CACHE_TIME) {
            return cached.records;
        }
    }
    if (updateInProgress.has(id)) {
        return recordsCache.has(id) ? recordsCache.get(id).records : [];
    }
    updateInProgress.add(id);
    try {
        // Используем bulkCheckIds для одного ID
        const results = await bulkCheckIds([id]);
        // console.log('[fetchRecordsForId] Ответ bulkCheckIds:', results);
        // console.log('[fetchRecordsForId] Ключи results:', Object.keys(results));
        // console.log('[fetchRecordsForId] id:', id);
        let allRecords = [];
        // Нормализация ключа
        const key = Object.keys(results).find(k => k == id);
        if (key && results[key]) {
            if (Array.isArray(results[key].records)) {
                allRecords = results[key].records;
            } else if (Array.isArray(results[key])) {
                allRecords = results[key];
            }
        }
        // console.log('[fetchRecordsForId] Итоговые записи для подсветки:', allRecords);
        recordsCache.set(id, {
            records: allRecords,
            timestamp: now
        });
        return allRecords;
    } catch (error) {
        console.error("❌ Критическая ошибка загрузки записей:", error);
        return [];
    } finally {
        updateInProgress.delete(id);
    }
}

// Одиночная отправка через bulkAddIds
async function sendIdToServer(id, apiKey) {
    try {
        const success = await bulkAddIds([id], apiKey);
        return success;
    } catch (e) {
        console.error("Ошибка отправки ID:", id, e);
        return false;
    }
}

// ==================== UI КОМПОНЕНТЫ ====================

function createControlButtons(id, initialRecords = []) {
    const container = document.createElement("div");
    container.className = "universal-id-controls";
    Object.assign(container.style, {
        position: "absolute",
        top: "4px",
        right: "4px",
        display: "flex",
        gap: "4px",
        zIndex: "99999",
        background: "rgba(255,255,255,0.9)",
        padding: "4px",
        borderRadius: "6px",
        border: "1px solid #ddd"
    });

    const copyBtn = document.createElement("button");
    copyBtn.textContent = "📋";
    copyBtn.title = `Копировать ID: ${id}`;
    Object.assign(copyBtn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: "1px solid #99ccee",
        borderRadius: "4px",
        backgroundColor: "#e0f0ff",
        color: "#003366"
    });
    
    copyBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        try {
            await navigator.clipboard.writeText(id);
            copyBtn.textContent = "✅";
            setTimeout(() => copyBtn.textContent = "📋", 1500);
        } catch {
            alert("Не удалось скопировать ID");
        }
    });

    const uploadBtn = document.createElement("button");
    uploadBtn.className = "universal-upload-btn";
    updateUploadButton(uploadBtn, initialRecords, id);

    uploadBtn.addEventListener("click", async (e) => {
        e.preventDefault();
        e.stopPropagation();
        
        if (uploadBtn._busy) return;
        uploadBtn._busy = true;
        
        uploadBtn.textContent = "⏳";
        uploadBtn.style.backgroundColor = "#fff3cd";
        uploadBtn.style.color = "#6a4f00";
        uploadBtn.disabled = true;

        const apiKey = await getApiKey();
        if (!apiKey) {
            alert("⚠️ API-ключ не задан. Кликните по иконке расширения и введите ключ.");
            uploadBtn._busy = false;
            updateUploadButton(uploadBtn, initialRecords, id);
            return;
        }

        const success = await sendIdToServer(id, apiKey);
        if (success) {
            recordsCache.delete(id);
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                applyHighlight(uploadBtn.closest(getCurrentSelectors()), shouldHighlight(records));
                uploadBtn._busy = false;
            }, 1000);
        } else {
            uploadBtn.textContent = "❌";
            setTimeout(async () => {
                const records = await fetchRecordsForId(id);
                updateUploadButton(uploadBtn, records, id);
                uploadBtn._busy = false;
            }, 1500);
        }
    });

    container.appendChild(copyBtn);
    container.appendChild(uploadBtn);
    return container;
}

function updateUploadButton(btn, records, id) {
    const hasRecords = Array.isArray(records) && records.length > 0;
    
    btn.textContent = hasRecords ? "☁️✔️" : "☁️";
    btn.disabled = false;
    
    Object.assign(btn.style, {
        padding: "4px 8px",
        fontSize: "12px",
        cursor: "pointer",
        border: hasRecords ? "1px solid #77cc88" : "1px solid #99cc99",
        borderRadius: "4px",
        backgroundColor: hasRecords ? "#2ecc71" : "#e9f9e9",
        color: hasRecords ? "white" : "#115511"
    });

    if (hasRecords) {
        // УЛУЧШЕННАЯ СТАТИСТИКА С УЧЕТОМ ГРУПП
        const recordsBySite = {};
        records.forEach(r => {
            const site = r.site || r.sourceSite || 'неизвестно';
            if (!recordsBySite[site]) {
                recordsBySite[site] = [];
            }
            recordsBySite[site].push(r);
        });

        let title = `ID: ${id}\n\n`;
        
        // Разделяем на текущий сайт и сайты из групп
        const currentSiteRecords = recordsBySite[SITE_NAME] || [];
        const groupSiteRecords = Object.entries(recordsBySite)
            .filter(([site]) => site !== SITE_NAME)
            .sort(([a], [b]) => a.localeCompare(b));
        
        // Текущий сайт
        if (currentSiteRecords.length > 0) {
            title += `📍 ТЕКУЩИЙ САЙТ (${SITE_NAME}): ${currentSiteRecords.length} записей\n`;
            currentSiteRecords.slice(0, 2).forEach(r => {
                const user = r.added_by || 'неизвестно';
                const time = new Date(r.created_at).toLocaleString('ru-RU', {
                    timeZone: 'Europe/Moscow',
                    day: "2-digit",
                    month: "2-digit", 
                    year: "numeric",
                    hour: "2-digit",
                    minute: "2-digit"
                });
                title += `   └ ${user} (${time})\n`;
            });
            if (currentSiteRecords.length > 2) {
                title += `   └ ... и ещё ${currentSiteRecords.length - 2} записей\n`;
            }
            title += '\n';
        }
        
        // Сайты из групп
        if (groupSiteRecords.length > 0) {
            title += `👥 НАЙДЕНО В ГРУППАХ:\n`;
            groupSiteRecords.forEach(([site, siteRecords]) => {
                title += `📍 ${site}: ${siteRecords.length} записей\n`;
                
                siteRecords.slice(0, 2).forEach(r => {
                    const user = r.added_by || 'неизвестно';
                    const time = new Date(r.created_at).toLocaleString('ru-RU', {
                        timeZone: 'Europe/Moscow',
                        day: "2-digit",
                        month: "2-digit", 
                        year: "numeric",
                        hour: "2-digit",
                        minute: "2-digit"
                    });
                    title += `   └ ${user} (${time})\n`;
                });
                
                if (siteRecords.length > 2) {
                    title += `   └ ... и ещё ${siteRecords.length - 2} записей\n`;
                }
                title += '\n';
            });
        }

        // ОБЩАЯ СТАТИСТИКА
        const totalCount = records.length;
        const currentSiteCount = currentSiteRecords.length;
        const groupsCount = totalCount - currentSiteCount;
        
        title += `📊 СВОДКА:\n`;
        title += `   Всего записей: ${totalCount}\n`;
        title += `   На этом сайте: ${currentSiteCount}\n`;
        title += `   В группах: ${groupsCount}\n`;
        title += `   Проверено сайтов: ${Object.keys(recordsBySite).length}\n\n`;
        title += `🖱️ Кликните, чтобы добавить снова`;

        btn.title = title;
        
        // Изменяем цвет кнопки если найдено в группах
        if (groupsCount > 0) {
            btn.style.backgroundColor = "#f39c12"; // оранжевый для групп
            btn.style.color = "white";
            btn.style.border = "1px solid #e67e22";
        }
    } else {
        btn.title = `Добавить ID ${id} на сервер`;
    }
}

// ==================== УВЕДОМЛЕНИЯ ====================

function showNotification(message) {
    const notification = document.createElement('div');
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #10b981;
        color: white;
        padding: 12px 16px;
        border-radius: 8px;
        z-index: 10000;
        font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
        font-size: 14px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        max-width: 300px;
        word-wrap: break-word;
    `;
    
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 4000);
}

// ==================== ОСНОВНАЯ ЛОГИКА ====================

function enhanceCard(el) {
    if (!el || !el.nodeType || el.nodeType !== 1 || el._universalEnhanced) {
        return;
    }
    
    try {
        // ПРОВЕРЯЕМ СЕЛЕКТОРЫ ПАРСЕРОВ - если элемент не подходит ни под один селектор, пропускаем
        if (!elementMatchesAnyParserSelector(el)) {
            return;
        }
        
        const id = extractIDFromElement(el);
        if (!id) {
            return;
        }

        // console.log(`✅ Найден ВАЛИДНЫЙ ID: ${id}`, el);
        el._universalEnhanced = true;
        el._lastEnhancedTime = Date.now(); // Добавляем временную метку
        
        // Устанавливаем позицию
        try {
            const style = window.getComputedStyle(el);
            if (style.position === "static") {
                el.style.position = "relative";
            }
        } catch (styleError) {
            // Игнорируем ошибки позиционирования
        }

        // Удаляем старые контролы ТОЛЬКО если они наши
        const oldControls = el.querySelector(".universal-id-controls");
        if (oldControls && oldControls._isUniversalControl) {
            oldControls.remove();
        }

        const controls = createControlButtons(id);
        controls._isUniversalControl = true; // Помечаем наши контролы
        el.appendChild(controls);

        // Обновляем статус
        updateCardStatus(id, el);
        
    } catch (error) {
        console.error('❌ Ошибка в enhanceCard:', error);
    }
}

async function updateCardStatus(id, element) {
    if (updateInProgress.has(id)) {
        return;
    }
    
    try {
        const records = await fetchRecordsForId(id);
        
        const btn = element.querySelector(".universal-upload-btn");
        if (btn) {
            updateUploadButton(btn, records, id);
        }
        
        applyHighlight(element, shouldHighlight(records));
    } catch (error) {
        console.warn("❌ Ошибка обновления статуса карточки:", error);
    }
}

// ==================== НАБЛЮДАТЕЛЬ И ИНИЦИАЛИЗАЦИЯ ====================


// Обновление: пакетная обработка раз в минуту (меньше запросов, пакетные bulk вызовы)
setInterval(async () => {
    try {
        // Не работать если вкладка не активна
        if (typeof document !== 'undefined' && document.hidden) return;

        await loadSiteSettings();
        await loadSiteParsers(); // обновляет селекторы

        const currentSelectors = getCurrentSelectors();
        if (currentSelectors === '.universal-id-no-elements-found') return;

        const cards = Array.from(document.querySelectorAll(currentSelectors));
        const idsMap = new Map();
        for (const el of cards) {
            const id = extractIDFromElement(el);
            if (id) {
                if (!idsMap.has(id)) idsMap.set(id, []);
                idsMap.get(id).push(el);
            }
        }

        const ids = Array.from(idsMap.keys());
        if (ids.length === 0) return;

        // Запрос единым bulk-вызовом
        const results = await bulkCheckIds(ids);

        // Обновляем кэш и DOM на основании ответа
        const now = Date.now();
        for (const id of ids) {
            const resForId = results[id] || results[id.toString()] || [];
            const normalized = Array.isArray(resForId.records) ? resForId.records : (Array.isArray(resForId) ? resForId : []);
            recordsCache.set(id, { records: normalized, timestamp: now });
            const elements = idsMap.get(id) || [];
            for (const el of elements) {
                const btn = el.querySelector('.universal-upload-btn');
                if (btn) updateUploadButton(btn, normalized, id);
                applyHighlight(el, shouldHighlight(normalized));
            }
        }
    } catch (e) {
        console.warn('Ошибка в пакетном обновлении карточек:', e);
    }
}, 60 * 1000); // 60s

// Обновленная обработка SPA навигации
// УЛУЧШЕННАЯ обработка SPA навигации с debounce
let spaNavigationTimeout = null;

function handleUrlChange() {
    if (spaNavigationTimeout) {
        clearTimeout(spaNavigationTimeout);
    }
    
    spaNavigationTimeout = setTimeout(() => {
        const newUrl = window.location.href;
        if (newUrl !== currentUrl) {
            currentUrl = newUrl;
            
            // Очищаем кэш и флаги с debounce
            recordsCache.clear();
            updateInProgress.clear();
            lastParsersLoad = 0;
            lastSettingsLoad = 0;
            
            // ПОЛНОЕ пересканирование с задержкой
            setTimeout(async () => {
                // Полностью сбрасываем все флаги
                document.querySelectorAll(getCurrentSelectors()).forEach(el => {
                    delete el._universalEnhanced;
                });
                
                // Удаляем ВСЕ старые контролы
                document.querySelectorAll('.universal-id-controls').forEach(control => {
                    if (control._isUniversalControl) {
                        control.remove();
                    }
                });
                
                await loadSiteSettings();
                await loadSiteParsers();
                
                // Полное пересканирование
                const currentSelectors = getCurrentSelectors();
                if (currentSelectors !== '.universal-id-no-elements-found') {
                    const cards = document.querySelectorAll(currentSelectors);
                    // Пакетная обработка для лучшей производительности
                    const batchSize = 20;
                    for (let i = 0; i < cards.length; i += batchSize) {
                        const batch = Array.from(cards).slice(i, i + batchSize);
                        batch.forEach(enhanceCard);
                        await new Promise(resolve => setTimeout(resolve, 50));
                    }
                }
                
                scanPageWithParsers();
            }, 3000); // Больше времени для загрузки страницы
        }
    }, 500); // Debounce 500ms
}


// Перехват SPA навигации
const originalPushState = history.pushState;
const originalReplaceState = history.replaceState;

history.pushState = function(...args) {
    originalPushState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

history.replaceState = function(...args) {
    originalReplaceState.apply(this, args);
    setTimeout(handleUrlChange, 100);
};

window.addEventListener('popstate', handleUrlChange);

// ==================== КОММУНИКАЦИЯ С POPUP ====================

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    // console.log('📨 Content script получил сообщение:', request);
    
    if (request.action === 'getParsersInfo') {
        sendResponse({
            success: true,
            parsers: siteParsers,
            site: SITE_NAME,
            settings: siteSettings,
            parsersCount: siteParsers.length,
            selectors: getCurrentSelectors(),
            selectorsCount: DYNAMIC_SELECTORS.split(', ').length
        });
        return true;
    }
    
    if (request.action === 'refreshParsers') {
        lastParsersLoad = 0;
        loadSiteParsers().then(() => {
            sendResponse({ 
                success: true,
                parsers: siteParsers,
                parsersCount: siteParsers.length,
                selectors: getCurrentSelectors()
            });
        });
        return true;
    }
    
    if (request.action === 'getSelectorsInfo') {
        sendResponse({
            selectors: getCurrentSelectors(),
            selectorsList: DYNAMIC_SELECTORS.split(', '),
            parsersWithSelectors: siteParsers.filter(p => p.selectors && p.selectors.length > 0)
        });
        return true;
    }
    
    if (request.action === 'rescanPage') {
        recordsCache.clear();
        rescanPageWithNewSelectors();
        sendResponse({ success: true });
        return true;
    }
    
    if (request.action === 'getSiteInfo') {
        sendResponse({
            site: SITE_NAME,
            settings: siteSettings
        });
        return true;
    }
    
    if (request.action === 'refreshSettings') {
        lastSettingsLoad = 0;
        lastParsersLoad = 0;
        recordsCache.clear();
        loadSiteSettings();
        loadSiteParsers();
        sendResponse({ success: true });
        return true;
    }
});

// ==================== ОТЛАДОЧНЫЕ ФУНКЦИИ ====================

// Функции для отладки селекторов
function debugSelectors() {
    // console.log('🔍 ДЕБАГ СЕЛЕКТОРОВ:');
    // console.log('Текущие селекторы:', getCurrentSelectors());
    // console.log('Парсеры с селекторами:', siteParsers.filter(p => p.selectors && p.selectors.length > 0));
    
    const elements = document.querySelectorAll(getCurrentSelectors());
    // console.log(`Найдено элементов: ${elements.length}`);
    
    elements.forEach((el, index) => {
        // console.log(`Элемент ${index + 1}:`, el);
        // console.log(`  HTML:`, el.outerHTML.substring(0, 200));
        
        // Проверяем какие селекторы совпадают
        const matchingSelectors = [];
        siteParsers.forEach(parser => {
            if (parser.selectors) {
                parser.selectors.forEach(selector => {
                    try {
                        if (el.matches(selector)) {
                            matchingSelectors.push(`${parser.name}: ${selector}`);
                        }
                    } catch (e) {
                        // Игнорируем ошибки селекторов
                    }
                });
            }
        });
        
        // console.log(`  Совпадающие селекторы:`, matchingSelectors);
    });
}

// Сделаем доступным для отладки
window.debugSelectors = debugSelectors;


// ==================== ИСПРАВЛЕННАЯ ИНИЦИАЛИЗАЦИЯ ====================

// Запуск
(async function init() {
    // console.log("🚀 Universal ID Collector запущен (полностью адаптирован под модульный сервер)");
    
    // Загружаем данные
    await Promise.all([loadSiteSettings(), loadSiteParsers()]);
    
    // Инициализируем наблюдатель только если есть селекторы
    const currentSelectors = getCurrentSelectors();
    if (currentSelectors !== '.universal-id-no-elements-found') {
        // Создаем наблюдатель (используем переменную observer, объявленную в начале)
        observer = new MutationObserver((mutations) => {
            const currentSelectors = getCurrentSelectors();
            
            // Если селекторы пустые, не обрабатываем мутации
            if (currentSelectors === '.universal-id-no-elements-found') {
                return;
            }
            
            mutations.forEach(mutation => {
                mutation.addedNodes.forEach(node => {
                    if (node.nodeType === 1) {
                        if (node.matches && node.matches(currentSelectors) && !node._universalEnhanced) {
                            setTimeout(() => enhanceCard(node), 100);
                        }
                        if (node.querySelectorAll) {
                            node.querySelectorAll(currentSelectors).forEach(el => {
                                if (!el._universalEnhanced) {
                                    setTimeout(() => enhanceCard(el), 100);
                                }
                            });
                        }
                    }
                });
            });
        });
        
        observer.observe(document.body, { childList: true, subtree: true });
        
        // Обрабатываем существующие карточки
        try {
            const cards = document.querySelectorAll(currentSelectors);
            // console.log(`🎯 Найдено ${cards.length} элементов для обработки`);
            cards.forEach(enhanceCard);
        } catch (error) {
            console.warn('⚠️ Ошибка при обработке существующих карточек:', error);
        }
    } else {
        // console.log('⚠️ Нет активных селекторов, наблюдатель не запущен');
    }
    
    // Резервное обновление (редко) — минимизируем нагрузку
    setInterval(async () => {
        try {
            if (typeof document !== 'undefined' && document.hidden) return;
            await loadSiteSettings();
            await loadSiteParsers();
        } catch (e) {
            console.warn('Ошибка в резервном обновлении парсеров/настроек:', e);
        }
    }, 5 * 60 * 1000); // 5 минут
    
    // Подключаемся к SSE для моментальных обновлений (если сервер поддерживает)
    (function initSSE(){
        try {
            const sseUrl = `${SERVER_URL}/api/ids/sse?site=${encodeURIComponent(SITE_NAME)}`;
            let retry = 2000;
            function connect() {
                const es = new EventSource(sseUrl);
                es.onmessage = async (e) => {
                    try {
                        const payload = JSON.parse(e.data);
                        if (payload && payload.event === 'id_added' && payload.data && payload.data.id) {
                            const id = String(payload.data.id);
                            recordsCache.delete(id);
                            // Обновляем только элементы с этим ID
                            const els = document.querySelectorAll(getCurrentSelectors());
                            for (const el of els) {
                                const found = extractIDFromElement(el);
                                if (found && String(found) === id) {
                                    const recs = await fetchRecordsForId(id);
                                    const btn = el.querySelector('.universal-upload-btn');
                                    if (btn) updateUploadButton(btn, recs, id);
                                    applyHighlight(el, shouldHighlight(recs));
                                }
                            }
                        }
                    } catch (err) {
                        console.warn('SSE parse error', err);
                    }
                };
                es.onerror = () => {
                    es.close();
                    setTimeout(() => { retry = Math.min(60000, retry * 1.5); connect(); }, retry);
                };
            }
            connect();
        } catch (e) {
            console.warn('SSE init failed', e);
        }
    })();

})();